/*
 * Decompiled with CFR 0.152.
 */
package team.tnt.collectoralbum.util.datagen;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.mojang.datafixers.util.Pair;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import team.tnt.collectoralbum.common.init.ItemRegistry;

public class AssetsGen {
    private static final String NAMESPACE = "collectorsalbum";
    private static final File DIR_ASSETS = new File("../src/main/resources/assets/collectorsalbum");
    private static final File DIR_MODELS = new File(DIR_ASSETS, "models");
    private static final File DIR_BLOCKSTATES = new File(DIR_ASSETS, "blockstates");
    private static final File DIR_MODELS_BLOCK = new File(DIR_MODELS, "block");
    private static final File DIR_MODELS_ITEM = new File(DIR_MODELS, "item");
    private static final File DIR_EXPORTED_LOCALIZATION = new File("./exported/datagen");
    private static final Logger LOG = LogManager.getLogger((String)"AssetsGenerator");
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private final List<Entry> datagenEntries = new ArrayList<Entry>();
    private final List<Pair<String, String>> localizationBuffer = new ArrayList<Pair<String, String>>();
    private final List<Runnable> finalActionsBuffer = new ArrayList<Runnable>();

    public static void main(String[] args) {
        long startTime = System.currentTimeMillis();
        try {
            new AssetsGen().run();
        }
        catch (Throwable t) {
            LOG.fatal((Object)t);
        }
        long totalExecutionTime = System.currentTimeMillis() - startTime;
        LOG.info("Data generation finished after {}ms", (Object)totalExecutionTime);
    }

    private void run() throws Exception {
        LOG.info("Running data generator");
        LOG.info("Registering data generator entries");
        this.registerEntry(ItemRegistry.class, this::runItemGen);
        LOG.info("Processing datagen entries");
        this.datagenEntries.parallelStream().forEach(entry -> {
            Class<?> type = entry.type();
            Field[] fields = type.getFields();
            String[] expectedIds = (String[])((Stream)Arrays.stream(fields).parallel()).filter(field -> {
                int modifiers = field.getModifiers();
                return Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers);
            }).map(field -> field.getName().toLowerCase(Locale.ROOT)).toArray(String[]::new);
            IDataGenerator generator = entry.generator();
            try {
                generator.generate(expectedIds);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        });
        LOG.info("Exporting prepared localizations");
        if (!this.localizationBuffer.isEmpty()) {
            LOG.info("Sorting localizations");
            this.localizationBuffer.sort(Comparator.comparing(Pair::getFirst));
            LOG.info("Prepared localizations dump");
            this.localizationBuffer.forEach(pair -> LOG.info("Key: {}, Value: {}", pair.getFirst(), pair.getSecond()));
            ZonedDateTime zdt = ZonedDateTime.now(ZoneOffset.UTC);
            String fileSuffix = "_" + zdt.format(DateTimeFormatter.BASIC_ISO_DATE);
            File file = new File(DIR_EXPORTED_LOCALIZATION, "en_us_exported" + fileSuffix + ".json");
            if (DIR_EXPORTED_LOCALIZATION.mkdirs()) {
                LOG.info("Created directory structure for localization exports");
            }
            if (file.createNewFile()) {
                LOG.info("Created new empty localizations file with name {}", (Object)file.getName());
            }
            LOG.info("Converting localizations to JSON format");
            JsonObject root = new JsonObject();
            for (Pair<String, String> pair2 : this.localizationBuffer) {
                root.addProperty((String)pair2.getFirst(), (String)pair2.getSecond());
            }
            this.writeJsonContentToFile((JsonElement)root, file);
            LOG.info("Localizations exported");
        } else {
            LOG.warn("No localizations to export, aborting");
        }
        LOG.info("Running post run actions and logging");
        this.finalActionsBuffer.forEach(Runnable::run);
        LOG.info("Datagen run has finished");
    }

    private void registerEntry(Class<?> type, IDataGenerator generator) {
        this.datagenEntries.add(new Entry(type, generator));
        LOG.info("Registered generator for {} class", (Object)type.getSimpleName());
    }

    private void runBlockGen(String[] objectIds) throws Exception {
        String localizationPrefix = String.format("block.%s.", NAMESPACE);
        int blockGenerated = 0;
        for (String id : objectIds) {
            LOG.info("Running datagen for block {}", (Object)id);
            File blockstateFile = new File(DIR_BLOCKSTATES, id + ".json");
            if (blockstateFile.exists()) {
                LOG.warn("Skipping datagen for block {}, file already exists", (Object)id);
                continue;
            }
            File blockModelFile = new File(DIR_MODELS_BLOCK, id + ".json");
            File itemModelFile = new File(DIR_MODELS_ITEM, id + ".json");
            if (blockstateFile.createNewFile()) {
                LOG.info("Created new empty blockstate file for block {}", (Object)id);
            }
            if (blockModelFile.createNewFile()) {
                LOG.info("Created new empty block model file for {}", (Object)id);
            }
            if (itemModelFile.createNewFile()) {
                LOG.info("Created new empty block item model file for {}", (Object)id);
            }
            LOG.info("Generating JSON contents for {} block", (Object)id);
            JsonObject blockstateJsonRoot = new JsonObject();
            JsonObject variantsJson = new JsonObject();
            JsonObject variantModelMappingJson = new JsonObject();
            variantModelMappingJson.addProperty("model", String.format("%s:block/%s", NAMESPACE, id));
            variantsJson.add("", (JsonElement)variantModelMappingJson);
            blockstateJsonRoot.add("variants", (JsonElement)variantsJson);
            JsonObject blockModelRoot = new JsonObject();
            blockModelRoot.addProperty("parent", "block/cube_all");
            JsonObject blockModelTextures = new JsonObject();
            blockModelTextures.addProperty("all", String.format("%s:block/%s", NAMESPACE, id));
            blockModelRoot.add("textures", (JsonElement)blockModelTextures);
            JsonObject itemModelRoot = new JsonObject();
            itemModelRoot.addProperty("parent", String.format("%s:block/%s", NAMESPACE, id));
            LOG.info("Writing blockstate file content for {}", (Object)id);
            this.writeJsonContentToFile((JsonElement)blockstateJsonRoot, blockstateFile);
            LOG.info("Writing block model file content for {}", (Object)id);
            this.writeJsonContentToFile((JsonElement)blockModelRoot, blockModelFile);
            LOG.info("Writing item block model file content for {}", (Object)id);
            this.writeJsonContentToFile((JsonElement)itemModelRoot, itemModelFile);
            LOG.info("Model files successfully created for {} block", (Object)id);
            this.prepareLocalizationPair(localizationPrefix, id);
            ++blockGenerated;
        }
        String text = "Generated " + blockGenerated + " block model files and localizations";
        this.finalActionsBuffer.add(() -> LOG.info(text));
    }

    private void runItemGen(String[] objectIds) throws Exception {
        int itemsGenerated = 0;
        String localizationPrefix = String.format("item.%s.", NAMESPACE);
        for (String objectId : objectIds) {
            LOG.info("Running datagen for item {}", (Object)objectId);
            File modelFile = new File(DIR_MODELS_ITEM, objectId + ".json");
            if (modelFile.exists()) {
                LOG.warn("Skipping datagen for item {}, file already exists", (Object)objectId);
                continue;
            }
            if (modelFile.createNewFile()) {
                LOG.info("Created new empty item model file for item {}", (Object)objectId);
            }
            LOG.info("Generating JSON contents for {} item", (Object)objectId);
            JsonObject rootJson = new JsonObject();
            rootJson.addProperty("parent", "item/generated");
            JsonObject texturesJson = new JsonObject();
            texturesJson.addProperty("layer0", String.format("%s:item/%s", NAMESPACE, objectId));
            rootJson.add("textures", (JsonElement)texturesJson);
            LOG.info("Populating model file of {} item", (Object)objectId);
            this.writeJsonContentToFile((JsonElement)rootJson, modelFile);
            LOG.info("Model file successfully created for {} item", (Object)objectId);
            this.prepareLocalizationPair(localizationPrefix, objectId);
            ++itemsGenerated;
        }
        String text = "Generated " + itemsGenerated + " item files and localizations";
        this.finalActionsBuffer.add(() -> LOG.info(text));
    }

    private void prepareLocalizationPair(String prefix, String objectId) {
        Pair localizationPair = new Pair((Object)(prefix + objectId), (Object)this.guessLocalization(objectId));
        this.localizationBuffer.add((Pair<String, String>)localizationPair);
        LOG.info("Prepared localization pair: " + localizationPair);
    }

    private String guessLocalization(String objectId) {
        return String.join((CharSequence)" ", (CharSequence[])Arrays.stream(objectId.split("_+")).map(word -> word.substring(0, 1).toUpperCase() + word.substring(1).toLowerCase()).toArray(String[]::new));
    }

    private void writeJsonContentToFile(JsonElement content, File dest) throws IOException {
        try (FileWriter fw = new FileWriter(dest);){
            fw.write(GSON.toJson(content));
        }
    }

    @FunctionalInterface
    private static interface IDataGenerator {
        public void generate(String[] var1) throws Exception;
    }

    private record Entry(Class<?> type, IDataGenerator generator) {
    }
}

